capture log close
clear all
macro drop _all
cd "/Volumes/Mike2025/MC_CLIENTS/2025/AER 2025/Stata/Time Varying Models"
set more off
set maxiter 3000

**** SET PERIOD ***
local perd "long"      // can be "long" or "short"

log using "Output/log/SFA-fecss/anOpexReg24-fecss-`perd'", replace text

/*--------------------------------------------------------------------------
Project: Quantonomics DNSP Opex Cost Function -  Time-Varying Models
		 Model fecss
Author:  Alice Giovani & Michael Cunningham
File:    anOpexReg24-fecss.do
Initial Date:    19-Mar-2025
Revised Date:	 1-Aug-2025

--------------------------------------------------------------------------*/

* Period & dependent variable
if "`perd'" == "long" {
	local startyr 2006
	}
else if "`perd'" == "short" {
	local startyr 2012
}
local endyr   2023

* 
scalar define ppp_aust   		   = 1.527404363
scalar define ppp_nz    		   = 1.482877481
scalar define ppp_canada		   = 1.249076963

scalar define propex_rebase_aust   = 1.238501756
scalar define propex_rebase_nz     = 1.219668445
scalar define propex_rebase_canada = 1.160126855  

* Read data
use Input/DNSPopex24, clear
xtset eiid year

gen int dnsp = eiid - 1000 if country == 1
qui tabulate dnsp, gen(d)
forvalues k = 1/13 {
	replace d`k' = 0 if d`k' == .
}

tabulate country, gen(jur)
correlate custnum circlen rmdem shareugc
tabulate year

summarize d* jur*

* Sample selection
*-------------------------
drop if country == 3 & year < (`startyr' - 1)
drop if country < 3 & year < `startyr'
drop if country == 3 & year > (`endyr' - 1)
drop if country < 3 & year > `endyr'

* Modify & create variables
*--------------------------

* Adressing change of VIC's DNSP from calendar to financial year in 2021
generate yr=year
replace yr=yr+0.5 if dnsp==3 & year<2021
replace yr=yr+0.5 if dnsp==8 & year<2021
replace yr=yr+0.5 if dnsp==9 & year<2021
replace yr=yr+0.5 if dnsp==11 & year<2021
replace yr=yr+0.5 if dnsp==13 & year<2021
replace yr=yr-0.25 if country==2
replace yr=yr+0.5 if country==3

replace propex = propex / propex_rebase_aust    if country==1
replace propex = propex / propex_rebase_nz      if country==2
replace propex = propex / propex_rebase_canada  if country==3
replace propex = propex * ppp_nz / ppp_aust     if country==2
replace propex = propex * ppp_canada / ppp_aust if country==3

* Regression log variables
gen lvc = log(opex / propex)
gen ly1 = log(custnum)
gen ly2 = log(circlen)
gen ly3 = log(rmdem)
gen lz1 = log(shareugc)

* mean correct output data
qui reg ly1
predict mly1
replace ly1=ly1-mly1
qui reg ly2
predict mly2
replace ly2=ly2-mly2
qui reg ly3
predict mly3
replace ly3=ly3-mly3

* TLG variables
gen ly11 = (ly1^2)/2
gen ly12 = ly1*ly2
gen ly13 = ly1*ly3
gen ly22 = (ly2^2)/2
gen ly23 = ly2*ly3
gen ly33 = (ly3^2)/2

* Time Trend Variables
gen aus_yr = cond(country == 1,yr,0)
gen nz_yr = cond(country == 2,yr,0)
gen ont_yr = cond(country == 3,yr,0)

gen yearx = 0
replace yearx = year - 2004 if country == 3
replace yearx = year - 2005 if country < 3

qui tabulate yearx, generate(yr)

* Labels
label variable ly1  "ln(Custnum)=x1" 
label variable ly2  "ln(CircLen)=x2" 
label variable ly3  "ln(RMDemand)=x3" 
label variable lz1  "ln(ShareUGC)" 
label variable ly11 "x1*x1/2" 
label variable ly12 "x1*x2" 
label variable ly13 "x1*x3" 
label variable ly22 "x2*x2/2" 
label variable ly23 "x2*x3" 
label variable ly33 "x3*x3/2"
label variable yr   "Year" 
label variable jur2 "New Zealand" 
label variable jur3 "Ontario" 

* Documentation

putdocx begin
putdocx paragraph, style(Title)
putdocx text ("Time Varying fecss - `perd' period" )

* Regression models
*---------------------- 
	
*===============================================
* SFACD  
*================================================

qui reg lvc ly1 ly2 ly3 lz1 yr
matrix b0 = e(b)
*, ln(e(rmse)^2) , .1, .1

sfpanel lvc ly1 ly2 ly3 lz1 yr, model(fecss) cost svfrontier(b0) iterate(500)

estimates store SFACD

* STATS & TESTS
predict yhat1, xb
predict yhat2, u
gen yhat = yhat1 + yhat2	
gen eh = lvc - yhat
*estat ic

scalar k_sfacd = 7 		
qui corr lvc yhat
local psadjr2 = 1 - ((1 - r(rho)^2)*(r(N) - 1)/(r(N) - k_sfacd))	
display "SFACD Pseudo-adjusted R2 = " `psadjr2'

iqr eh
swilk eh
collin ly1 ly2 ly3 lz1 yr
linktest


* OUTPUT ELASTICITIES	
predictnl ely1 = (_b[ly1]), ci(lw_y1 up_y1)
predictnl ely2 = (_b[ly2]), ci(lw_y2 up_y2)
predictnl ely3 = (_b[ly3]), ci(lw_y3 up_y3)
predictnl elY  = (_b[ly1] + _b[ly2] + _b[ly3]), ci(lw_Y up_Y)

label var ely1 "ely1"
label var ely2 "ely2"
label var ely3 "ely3"
label var elY  "elY"
label var lw_y1 "lw_y1"
label var up_y1 "up_y1"
label var lw_y2  "lw_y2"
label var up_y2  "up_y2"
label var lw_y3  "lw_y3"
label var up_y3  "up_y3"
label var lw_Y  "lw_Y"
label var up_Y  "up_Y"

putdocx paragraph, style(Subtitle)
putdocx text ("SFACD-fecss-Elasticities - `perd' period")
table (country) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect

*EFFICIENCY SCORES

predict ineff, u 
gen eff = exp(-ineff) 
table (dnsp) (var), stat(mean eff)

*Charts	
twoway (line eff year if dnsp == 1,  sort lcolor(blue)      lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 2,  sort lcolor(red)       lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 3,  sort lcolor(green)     lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 4,  sort lcolor(orange)    lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 5,  sort lcolor(maroon)    lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 6,  sort lcolor(teal)      lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 7,  sort lcolor(magenta)   lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 8,  sort lcolor(dkgreen)   lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 9,  sort lcolor(purple)    lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 10, sort lcolor(brown)     lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 11, sort lcolor(navy)      lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 12, sort lcolor(pink)      lpattern(solid) 	  lwidth(0.5)) ///
       (line eff year if dnsp == 13, sort lcolor(olive)     lpattern(solid)       lwidth(0.5)), ///
       legend(order(1 "EVO" 2 "AGD" 3 "CIT" 4 "END" 5 "ENX" ///
                    6 "ERG" 7 "ESS" 8 "JEN" 9 "PCR" 10 "SAP" ///
                    11 "AND" 12 "TND" 13 "UED") title("DNSP")) ///
       xtitle("Year") ytitle("Efficiency Score") ///
       xlabel(`startyr'(1)`endyr', angle(90) grid)  /// 
       ylabel(0(0.1)1, grid) /// 
       xscale(range(`startyr' `endyr')) /// 
       yscale(range(0 1))       

graph export "Output/chart/SFA-fecss/SFACD-fecss-Eff-`perd'.png", replace
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-fecss/SFACD-fecss-Eff-`perd'.png", width(8) height(5.6)
  
estout  SFACD using "Output/xlsx/SFA-fecss/SFACD-fecss-Reg-`perd'.tsv", ///
	cells("b(fmt(4)) se(fmt(4)) t(fmt(4))") /// 
	style(tab) label title("SFACD-fecss-`perd'") stats(N bic, fmt(3)) replace
  
 drop eff ineff ely1 lw_y1 up_y1 ely2 lw_y2 up_y2 ely3 lw_y3 up_y3 ///
	   elY lw_Y up_Y yhat1 yhat2 yhat eh


*===================================================
*  SFATLG
*===================================================

qui reg lvc ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr
matrix b0 = e(b)
*, ln(e(rmse)^2) , .1, .1

sfpanel lvc ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr, model(fecss) ///
	cost svfrontier(b0) iterate(500)

estimates store SFATLG
	
* STATS TESTS
predict yhat1, xb
predict yhat2, u
gen yhat = yhat1 + yhat2	
gen eh = lvc - yhat

scalar k_sfatlg = 13		
qui corr lvc yhat
local psadjr2 = 1 - ((1 - r(rho)^2)*(r(N) - 1)/(r(N) - k_sfacd))	
display "SFATLG Pseudo-adjusted R2 = " `psadjr2'

iqr eh
swilk eh
collin ly1 ly2 ly3 ly11 ly12 ly13 ly22 ly23 ly33 lz1 yr
linktest

test ly11 ly12 ly13 ly22 ly23 ly33

* OUTPUT ELASTICITIES	

predictnl ely1 = (_b[ly1] + _b[ly11]*ly1 + _b[ly12]*ly2 + _b[ly13]*ly3), ///
	ci(lw_y1 up_y1) p(p1)
predictnl ely2 = (_b[ly2] + _b[ly12]*ly1 + _b[ly22]*ly2 + _b[ly23]*ly3), ///
	ci(lw_y2 up_y2) p(p2)
predictnl ely3 = (_b[ly3] + _b[ly13]*ly1 + _b[ly23]*ly2 + _b[ly33]*ly3), ///
	ci(lw_y3 up_y3) p(p3)
predictnl elY  = (_b[ly1]  + _b[ly11]*ly1 + _b[ly12]*ly2 + _b[ly13]*ly3 + ///
	_b[ly2] + _b[ly12]*ly1 + _b[ly22]*ly2 + _b[ly23]*ly3 + _b[ly3] + ///
	_b[ly13]*ly1 + _b[ly23]*ly2 + _b[ly33]*ly3), ci(lw_Y up_Y)

label var ely1 "ely1"
label var ely2 "ely2"
label var ely3 "ely3"
label var elY  "elY"
label var lw_y1 "lw_y1"
label var up_y1 "up_y1"
label var lw_y2  "lw_y2"
label var up_y2  "up_y2"
label var lw_y3  "lw_y3"
label var up_y3  "up_y3"
label var lw_Y  "lw_Y"
label var up_Y  "up_Y"

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG-fecss Elasticities - `perd' period")
table (country) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect
table (dnsp) (var) , stat(mean ely1 ely2 ely3 elY) nformat(%5.3f)
putdocx collect

display "MONOTONICITY VIOLATIONS"
gen mon1 = 0
gen mon2 = 0
gen mon3 = 0
gen montot = 0
replace mon1 = cond(ely1 < 0,100,0)	
replace mon2 = cond(ely2 < 0,100,0)	
replace mon3 = cond(ely3 < 0,100,0)
replace montot = 100 if mon1 == 100 | mon2 == 100 | mon3 == 100

putdocx pagebreak
putdocx paragraph, style(Subtitle)
putdocx text ("SFATLG-fecss Monotonicity Violations - `perd' period")
table (country) (var) , stat(mean mon1 mon2 mon3 montot) nformat(%5.1f)
putdocx collect
table (dnsp) (var) , stat(mean mon1 mon2 mon3 montot) nformat(%5.1f)
putdocx collect

*EFFICIENCY SCORES

predict ineff, u 
gen eff = exp(-ineff) 
table (dnsp) (var), stat(mean eff)

twoway (line eff year if dnsp == 1,  sort lcolor(blue)      lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 2,  sort lcolor(red)       lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 3,  sort lcolor(green)     lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 4,  sort lcolor(orange)    lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 5,  sort lcolor(maroon)    lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 6,  sort lcolor(teal)      lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 7,  sort lcolor(magenta)   lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 8,  sort lcolor(dkgreen)   lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 9,  sort lcolor(purple)    lpattern(solid)       lwidth(0.5)) ///
       (line eff year if dnsp == 10, sort lcolor(brown)     lpattern(dash)        lwidth(0.5)) ///
       (line eff year if dnsp == 11, sort lcolor(navy)      lpattern(longdash)    lwidth(0.5)) ///
       (line eff year if dnsp == 12, sort lcolor(pink)      lpattern(solid) 	  lwidth(0.5)) ///
       (line eff year if dnsp == 13, sort lcolor(olive)     lpattern(solid)       lwidth(0.5)), ///
       legend(order(1 "EVO" 2 "AGD" 3 "CIT" 4 "END" 5 "ENX" ///
                    6 "ERG" 7 "ESS" 8 "JEN" 9 "PCR" 10 "SAP" ///
                    11 "AND" 12 "TND" 13 "UED") title("DNSP")) ///
       xtitle("Year") ytitle("Efficiency Score") ///
       xlabel(`startyr'(1)`endyr', angle(90) grid)  /// 
       ylabel(0(0.1)1, grid) /// 
       xscale(range(`startyr' `endyr')) /// 
       yscale(range(0 1))       
      	   
graph export "Output/chart/SFA-fecss/SFATLG-fecss-Eff-`perd'.png", replace
putdocx paragraph, halign(center)
putdocx image "Output/chart/SFA-fecss/SFATLG-fecss-Eff-`perd'.png", ///
	width(8) height(5.6)
  
estout  SFATLG using "Output/xlsx/SFA-fecss/SFATLG-fecss-Reg-`perd'.tsv", ///
	cells("b(fmt(4)) se(fmt(4)) t(fmt(4))") /// 
	style(tab) label title("SFATLG-fecss-`perd'") stats(N bic, fmt(3)) replace
  

putdocx save "Output/docx/SFA-fecss/SFA-fecss-`perd'", replace

**
log close

